<?php

declare(strict_types=1);

namespace Drupal\browsersync;

use Drupal\Core\Theme\ThemeManagerInterface;

/**
 * Browsersync Helper class.
 */
class BrowsersyncHelper {

  /**
   * Browsersync default host.
   *
   * @link http://www.browsersync.io/docs/options/#option-host
   */
  const BROWSERSYNC_DEFAULT_HOST = 'HOST';

  /**
   * Browsersync default port.
   *
   * @link http://www.browsersync.io/docs/options/#option-port
   */
  const BROWSERSYNC_DEFAULT_PORT = '3000';

  /**
   * Constructs a BrowsersyncHelper object.
   *
   * @param \Drupal\Core\Theme\ThemeManagerInterface $themeManager
   *   The 'theme.manager' service.
   */
  public function __construct(
    private readonly ThemeManagerInterface $themeManager,
  ) {}

  /**
   * Retrieves a setting for the current theme or for a given theme.
   *
   * @param string $setting_name
   *   The name of the setting to be retrieved.
   * @param string|null $theme
   *   (optional) The name of a given theme. Defaults to the current theme.
   *
   * @return mixed
   *   The value of the requested setting.
   *
   * @see theme_get_setting()
   */
  public function browsersyncGetSetting(string $setting_name, ?string $theme = NULL): mixed {
    $cache = &drupal_static('theme_get_setting', []);

    // If no key is given, use the current theme if we can determine it.
    if (!isset($theme)) {
      $theme = $this->themeManager->getActiveTheme()->getName();
    }

    // Prefix the setting name with the module's namespace.
    $setting_name = 'third_party_settings.browsersync.' . $setting_name;

    if (empty($cache[$theme])) {
      // If the cache has not been filled yet, invoke theme_get_setting to
      // retrieve the value. This will populate the cache and make it available
      // for subsequent requests.
      $setting = theme_get_setting($setting_name, $theme);
    }
    else {
      // Retrieve the value from the cache.
      $setting = $cache[$theme]->get($setting_name);
    }

    return $setting;
  }

}
