# Elasticsearch - Search API

## About

This module provides a framework to set up custom Elasticsearch based search pages.

It depends on search_api to manage your indexed data and elasticsearch_connector to set up a connection
with your Elasticsearch backend.

The elasticsearch_search_api module provides services, interfaces, Controllers and Classes to help you set up
custom Elasticsearch search pages, including the option to add facets, modify the search query that is sent to Elastic, add different search suggesters and analysers to fields (for example: n-gram) or add different searching
'strategies' to your Elasticsearch index.

Out-of-the box the module provides the ability to add autosuggestion, did-you-mean functionality,
synonyms and a strategy to copy all your indexed field data into a 'custom_all' text based field.

The idea of this module is to use it as a framework when building your own custom search module.
There is a lot of functionality that is included in the base module, but in your own search implementation,
services, controllers, templates, etc can be extended or extra searching strategies can be added to fit your needs.

## Installation

Add both this module and the jquery loading overlay library to your project's composer.json, which is a dependency of this module:
```
"repositories": [
    {
      "type": "package",
      "package": {
        "name": "gasparesganga/jquery-loading-overlay",
        "version": "2.1.7",
        "type": "drupal-library",
        "dist": {
          "url": "https://github.com/gasparesganga/jquery-loading-overlay/archive/refs/tags/v2.1.7.zip",
          "type": "zip"
        }
      }
    }
  ]
```
And install it as usual: `composer require drupal/elasticsearch_search_api`

When enabling the module, the 'search_index' view mode will be added to all existing content types.
Remember to run config export afterwards, in order to persist these view mode settings.

### Example module elasticsearch_search_api_example

The base module has a submodule, elasticsearch_search_api_example. This submodule is intended as a demo
of a custom elasticsearch implementation or as a reference point when starting your own custom implementation.

#### Installation steps

* Create an elasticsearch cluster via **/admin/config/search/elasticsearch-connector/cluster/add**
* Create a search_api server via **/admin/config/search/search-api/add-server** with machine name "**elastic**"
* Enable the module "Elasticsearch - Search API Example"
* An empty search instance should now be available on **/example/search**.

#### Example content

To create content to populate the example search, add some "Elasticsearch page" nodes.
Facets are activated, they can be used by creating "page_type" taxonomy terms. These can be referenced on "Elasticsearch page" content.

## Usage

This module should be used as a starting point for your own custom search implementation. It provides services that handle the communication between
Drupal and the ElasticSearch backend.

You can take a look at the elasticsearch_search_api_example module as a reference on how to start building your own custom search module.

## Services

The base module has a services.yml.example file. Each custom search implementation should define its own services.yml file and define what services should be used
on their custom implementation.

#### elasticsearch_search_api.elasticsearch_params_builder

The params builder service builds the parameters for the search action that we send to the elasticsearch backend. This service builds an array that will be converted
to JSON and sent over to the elasticsearch backend.

JSON examples:

Default example query, when not using keywords:
```
GET elasticsearch_index_drupal_default/_search
{
  "from": 0,
  "size": 10,
  "query": {
    "bool": {
      "must": []
    }
  },
  "highlight": {
    "fields": {
      "title": {}
    },
    "pre_tags": [
      "<strong>"
    ],
    "post_tags": [
      "</strong>"
    ]
  }
}
```
Example query when using keywords:
```
GET elasticsearch_index_drupal_default/_search
{
  "from": 0,
  "size": 10,
  "query": {
    "bool": {
      "filter": {
        "bool": {
          "must": []
        }
      },
      "should": [
        {
          "query_string": {
            "query": "keyword",
            "fields": []
          }
        },
        {
          "nested": {
            "path": "es_attachment",
            "query": {
              "bool": {
                "must": {
                  "query_string": {
                    "query": "keyword",
                    "fields": [
                      "title^1.0"
                    ]
                  }
                }
              }
            }
          }
        }
      ],
      "minimum_should_match": 1
    }
  },
  "highlight": {
    "fields": {
      "es_attachment.attachment.content": {}
    },
    "pre_tags": [
      "<strong>"
    ],
    "post_tags": [
      "</strong>"
    ]
  }
}
```

#### elasticsearch_search_api.search_action_factory
Factory that builds a search action based on HTTP request query parameters.

This service is required by the SearchController.
```
$query = $request->query;
```
```
$searchAction = $this->searchActionFactory->searchActionFromQuery($query, $this->facets, $request->isXmlHttpRequest());
```

It is expected that the request has a 'keyword' parameter, this will be used as the search keyword that will be sent to elasticsearch.
```
  public function searchActionFromQuery(ParameterBag $query, array $facets, bool $isXmlHttpRequest): FacetedKeywordSearchAction {
    $keyword = $query->get('keyword');
```

If the elasticsearch implementation has a faceted search, the facet collection also needs to
be passed from the controller to the searchAction.

The SearchActionFactory also lets you set the amount of search results per page. This is a parameter that can be altered
in the services.yml file:
```
elasticsearch_search_api.search_page_size: 10
```

#### elasticsearch_search_api.elasticsearch_result_parser
This service parses a raw ElasticSearch response into a SearchResult object.
The SearchResult object is a value object that represents the search results returned by the ElasticSearch backend.

This service is used by the Controller.

#### elasticsearch_search_api.search_repository
This service is used by the Controller.

It is used to pass the parameters to the client of the elasticsearch cluster and load items
out of the index by id.

For example:
The JSON that is built by the elasticsearch_params_builder service is passed to the client through
the search_repository service.

```
$params = $this->searchParamsBuilder->build($searchAction);
$response = $this->searchRepository->query($params);
```
Or the query that is sent to return suggestions on the 'title' entity field:
```
    $params = [
      'body' => [
        '_source' => 'title',
        'suggest' => [
          'search-suggest' => [
            'prefix' => $searchQuery,
            'completion' => [
              'field' => 'search_suggest',
              'size' => 10,
            ],
          ],
        ],
      ],
    ];

    $response = $this->searchRepository->query($params);
```

#### elasticsearch_search_api.suggest.title_suggester
This service uses the search_repository service to add a suggester to the elasticsearch index settings.

This service adds a suggester for the 'title' field. It is required that your
index at least has a title field with the 'title' as property path in order for this to work.

More information about suggesters in ElasticSearch can be found on:
https://www.elastic.co/guide/en/elasticsearch/reference/current/search-suggesters.html

#### elasticsearch_search_api.term_facet_storage
Storage service to fetch metadata from Drupal taxonomy terms. Used for facet building.

#### elasticsearch_search_api.term_tree_storage
Storage service to fetch metadata from hierarchical Drupal taxonomy terms. Used for hierarchical facet building.

#### elasticsearch_search_api.event_subscriber.initialize_index
Event subscriber that triggers on the PREPARE_INDEX and PREPARE_INDEX_MAPPING events sent out by
the elasticsearch_connector module.

In the base module, this event subscriber is used to add an ngram tokenizer and analyser to the index
configuration.

By default, this added ngram analyzer is also added to the 'title' (entity title) field.

More info about n-gram tokenizers within ElasticSearch can be found on: https://www.elastic.co/guide/en/elasticsearch/reference/current/analysis-ngram-tokenizer.html

## Routes & Controllers

A search results page typically needs two routes/callbacks:
- search callback to render the search page on page loads
- filter callback to update the page using ajax

See `elasticsearch_search_api.routing.yml.example` for more info

For "default" search pages the `src/Controller/SearchController.php` can be extended and will provide most features. This can keep custom controllers relatively simple:
```
class MyCustomController extends SearchController {

  public function __construct(RendererInterface $renderer, CurrentRouteMatch $routeMatch, SearchActionFactory $searchActionFactory, ElasticSearchParamsBuilder $searchParamsBuilder, ElasticSearchResultParser $resultParser, SuggesterInterface $suggester, SearchRepository $searchRepository, EntityTypeManagerInterface $entityTypeManager, RequestStack $requestStack) {
    parent::__construct($renderer, $routeMatch, $searchActionFactory, $searchParamsBuilder, $resultParser, $suggester, $searchRepository, $entityTypeManager);

    $this->facets = ['my-custom-facet'];
  }

  public static function create(ContainerInterface $container) {
    return new static($container->get('renderer'), $container->get('current_route_match'), $container->get('elasticsearch_search_api.search_action_factory'), $container->get('elasticsearch_search_api.elasticsearch_params_builder'), $container->get('elasticsearch_search_api.elasticsearch_result_parser'), $container->get('elasticsearch_search_api.suggest.title_suggester'), $container->get('elasticsearch_search_api.search_repository'), $container->get('entity_type.manager'), $container->get('request_stack'));
  }

  protected function renderPager(ParameterBag $query, int $total, int $size) {
    return [
      '#theme' => 'elasticsearch_search_api_pager',
      '#tags' => [
        1 => 'Previous',
        3 => 'Next',
      ],
      '#element' => 0,
      '#parameters' => $query->all(),
      '#total_items' => $total,
      '#items_per_page' => $size,
      '#route_name' => $this->getSearchUrl()->getRouteName(),
    ];
  }

  protected function getSuggestions(ParameterBag $query) {
    return [];
  }

  protected function getSearchHeader() {
    return [
      '#type' => 'container',
      '#children' => $this->formBuilder()->getForm(SearchForm::class),
    ];
  }

  protected function getSearchUrl(): Url {
    return Url::fromRoute('my_module.my_search_route');
  }

  protected function getFilterUrl(): Url {
    return Url::fromRoute('my_module.my_filter_route');
  }

}
```

## Facets

Creating and using facets requires the following:
- an instance of `Drupal\elasticsearch_search_api\Search\Facet\Control\CompositeFacetControlInterface` or `Drupal\elasticsearch_search_api\Search\Facet\Control\FacetControlInterface`
- a service tagged in the following format `elasticsearch_search_api.facet_control.my_facet`
- adding the facet to the constructor of the controller (see above)

Example of term-based facet:
```
class RegionFacetControl extends TermFacetBase {

  use StringTranslationTrait;

  const VOCABULARY_ID = 'MY_VOCABULARY';
  const PROPERTY_PATH = 'SEARCH_API_PROPERTY_PATH';

  public function __construct(FacetValueMetaDataTreeStorageInterface $facetValueMetaDataTreeStorage, string $routeName, EntityTypeManagerInterface $entityTypeManager) {
    parent::__construct($facetValueMetaDataTreeStorage, $routeName, $entityTypeManager);

    $this->setVocabulary(self::VOCABULARY_ID);
    $this->setfacetValuesSortMethod(self::SORT_TERM_WEIGHT);
    $this->setCanSelectMultiple(FALSE);
  }

  public function getFieldName(): string {
    return self::PROPERTY_PATH;
  }

  public function addToAggregations(): bool {
    return TRUE;
  }

  protected function getFacetTitle() {
    return sprintf('<h2>%s</h2>', $this->t('MY_FACET_TITLE'));
  }
```

## Searching strategies

The base module has a SyncStrategy base class and a SyncService. These are used to add custom
searching strategies to your ElasticSearch index.

The SyncStrategy base class should be extended by your custom search strategies.
The SyncService is used to alter the index settings and field mapping to add these searching strategies
to your index.
```
  public function execute(ClientInterface $client, array $settingsParams = [], array $mappingParams = []) {
    try {
      $client->indices()->close(['index' => $this->indexName]);
      if (!empty($settingsParams)) {
        $client->indices()->putSettings($settingsParams);
      }
      if (!empty($mappingParams)) {
        $client->indices()->putMapping($mappingParams);
      }
      return TRUE;
    }
    catch (\Exception $e) {
      watchdog_exception('elasticsearch_search_api', $e);
      return FALSE;
    }

    finally {
      sleep(1);
      $client->indices()->open(['index' => $this->indexName]);
    }
  }
```


Within the module file there is a hook_cron implementation that will trigger the synchronization of the
searching strategies with the Elasticsearch index.

This will apply the changes to the index settings and field mappings, and trigger a reïndex of the data.
```
  public function sync() {
    // Update analysis.
    /** @var \Drupal\elasticsearch_search_api\SyncStrategyInterface $strategy */
    foreach ($this->strategies as $strategy) {
      $strategy->execute($this->client);
    }

    // Reindex all items, so synonyms are correctly picked up.
    $this->reindexItems();
  }
```

The base module provides 4 different searching strategies out-of-the-box:

#### Autosuggest
This strategy adds a completion field type named 'search_suggest' to the index.
By default this strategy will alter the indexed 'title' field to copy it's data over to this added
search_suggest field.

#### CustomAll
This strategy will add a 'custom_all' field of type 'text' to the index field mappings.
All supported fields will be altered to have their data copied to this custom_all field.

This strategy can be usefull if you do not want to configure the ParamsBuilder to query each field
individually but instead query the 'custom_all' field.

#### DidYouMean
This strategy adds trigram analyser to the field mappings (n-gram of size 3).
It is added to the title field mapping and can be used by the controller to render a 'Did you mean'
snippet.

#### Synonoyms
This strategy can be used to add a synonyms graph token filter to the index settings.
More info: https://www.elastic.co/guide/en/elasticsearch/reference/current/analysis-synonym-graph-tokenfilter.html#analysis-synonym-graph-tokenfilter

#### Keymatches
Keymatches can be used to pin search results to the top of a search page.
Implement the KeymatchEntryFactory and KeymatchService services and add a route for the Keymatch form.
