<?php

namespace Drupal\elasticsearch_search_api\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Provides a search form to launch a search request to the ElasticSearch index.
 */
class SearchForm extends FormBase {

  /**
   * Returns a unique string identifying the form.
   *
   * @return string
   *   The unique string identifying the form.
   */
  public function getFormId() {
    return 'elasticsearch_search_api_form';
  }

  /**
   * Form constructor.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param bool $add_inline_autocomplete
   *   Boolean to indicate if the autocomplete suffix should be added.
   * @param string|null $custom_keyword_id
   *   Custom id for the keyword field.
   * @param string|null $custom_form_id
   *   Custom id for the form.
   * @param bool $ajax_form
   *   Boolean indicating if this form should support ajax submits.
   * @param mixed $redirect_url
   *   Url to redirect to on form submit.
   *   A default will be set if none is provided.
   * @param array $additional_query_params
   *   Additional query parameters to include in the redirect.
   *
   * @return array
   *   The form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state, bool $add_inline_autocomplete = FALSE, ?string $custom_keyword_id = NULL, ?string $custom_form_id = NULL, bool $ajax_form = FALSE, $redirect_url = FALSE, array $additional_query_params = []) {
    if (!$redirect_url instanceof Url) {
      $redirect_url = Url::fromRoute('elasticsearch_search_api.search');
    }

    $form_state->set('redirectUrl', $redirect_url);
    $form_state->set('additionalQueryParams', $additional_query_params);
    $keyword = $this->getRequest()->get('keyword');

    if ($ajax_form) {
      $form['#attributes']['data-ajax-search-form'] = 1;
    }

    if (!is_null($custom_keyword_id)) {
      $form['#attributes']['id'] = $custom_form_id;
    }

    // Add additional query parameters as hidden values.
    foreach ($additional_query_params as $key => $value) {
      $form[$key] = [
        '#type' => 'hidden',
        '#attributes' => [
          'class' => ['hidden-query-param'],
        ],
        '#value' => $value,
      ];
    }

    $form['keyword'] = [
      '#type' => 'textfield',
      '#attributes' => [
        'placeholder' => $this->t('What are you searching for?'),
        'autocomplete' => 'off',
      ],
      '#default_value' => $keyword,
      '#prefix' => '<span class="glyphicon glyphicon-search" aria-hidden="true"></span>',
    ];

    if ($add_inline_autocomplete) {
      $form['keyword']['#suffix'] = '<div class="search-autocomplete-inline"></div>';
      $form['#attached']['library'][] = 'elasticsearch_search_api/inline-autocomplete';
      $form['#attached']['drupalSettings']['elasticsearch_search_api']['header_form_class'] = Html::getClass($this->getFormId());
      $form['#attached']['drupalSettings']['elasticsearch_search_api']['autocomplete_endpoint'] = $this->getAutocompleteUrl()->toString();
    }

    if (!is_null($custom_keyword_id)) {
      $form['keyword']['#id'] = $custom_keyword_id;
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Search'),
      '#id' => 'edit-submit-search-submit',
    ];

    return $form;
  }

  /**
   * Form submission handler.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\Core\Url $url */
    $url = $form_state->get('redirectUrl');
    $queryParams = $form_state->get('additionalQueryParams');
    $queryParams['keyword'] = $form_state->getValue('keyword');

    $form_state->setRedirect($url->getRouteName(), $url->getRouteParameters(), [
      'query' => $queryParams,
    ]);
  }

  /**
   * Get the autocomplete url.
   *
   * This function must be used to assign autocomplete route info
   * in all classes that inherit this base class, since the route name / params
   * will differ for each search instance.
   *
   * @return \Drupal\Core\Url
   *   The autocomplete route url.
   */
  protected function getAutocompleteUrl(): Url {
    return Url::fromRoute('elasticsearch_search_api.autocomplete');
  }

}
