<?php

namespace Drupal\elasticsearch_search_api;

use Drupal\Core\Utility\Error;
use Drupal\elasticsearch_connector\ElasticSearch\Parameters\Factory\IndexFactory;
use Drupal\search_api\Entity\Index;
use nodespark\DESConnector\ClientInterface;

/**
 * SyncStrategy base class.
 *
 * @package Drupal\elasticsearch_search_api
 */
class SyncStrategy implements SyncStrategyInterface {

  /**
   * Index.
   *
   * @var \Drupal\search_api\Entity\Index
   */
  protected $index;

  /**
   * The index name.
   *
   * @var string
   */
  protected $indexName;

  /**
   * SyncStrategy constructor.
   *
   * @param \Drupal\search_api\Entity\Index $index
   *   The index object.
   */
  public function __construct(Index $index) {
    $this->index = $index;
    $this->indexName = IndexFactory::getIndexName($this->index);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(ClientInterface $client, array $settingsParams = [], array $mappingParams = []) {
    try {
      $client->indices()->close(['index' => $this->indexName]);
      if (!empty($settingsParams)) {
        $client->indices()->putSettings($settingsParams);
      }
      if (!empty($mappingParams)) {
        $client->indices()->putMapping($mappingParams);
      }
      return TRUE;
    }
    catch (\Exception $e) {
      Error::logException(\Drupal::logger('elasticsearch_search_api'), $e);
      return FALSE;
    }

    finally {
      sleep(1);
      $client->indices()->open(['index' => $this->indexName]);
    }
  }

  /**
   * Returns the index field mapping request.
   *
   * @param \nodespark\DESConnector\ClientInterface $client
   *   The client.
   * @param string $fields
   *   A comma-separated list of fields.
   *
   * @return false|mixed
   *   Returns the response from the client, or false if the request failed.
   */
  public function getFieldMapping(ClientInterface $client, $fields) {
    $params = [
      'index' => $this->indexName,
      'fields' => $fields,
    ];

    $response = $client->indices()->getFieldMapping($params);
    if (!empty($response)) {
      // Strip the index.
      return current($response);
    }
    else {
      return FALSE;
    }
  }

}
