<?php

namespace Drupal\moderation_note\Entity;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\moderation_note\ModerationNoteInterface;
use Drupal\user\UserInterface;

/**
 * Defines the moderation_note entity.
 *
 * @ContentEntityType(
 *   id = "moderation_note",
 *   label = @Translation("Moderation note"),
 *   handlers = {
 *     "access" = "Drupal\moderation_note\AccessControlHandler",
 *     "view_builder" = "Drupal\moderation_note\ModerationNoteViewBuilder",
 *     "form" = {
 *       "create" = "Drupal\moderation_note\ModerationNoteForm",
 *       "edit" = "Drupal\moderation_note\ModerationNoteForm",
 *       "reply" = "Drupal\moderation_note\ModerationNoteForm",
 *       "delete" = "Drupal\moderation_note\Form\ModerationNoteDeleteForm",
 *       "resolve" = "Drupal\moderation_note\Form\ModerationNoteResolveForm",
 *     },
 *     "views_data" = "Drupal\moderation_note\ModerationNoteViewsData",
 *   },
 *   base_table = "moderation_note",
 *   uri_callback = "moderation_note_uri",
 *   admin_permission = "administer moderation notes",
 *   fieldable = FALSE,
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "published" = "published"
 *   },
 * )
 */
class ModerationNote extends ContentEntityBase implements ModerationNoteInterface {

  use EntityChangedTrait;
  use EntityPublishedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'uid' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields += static::publishedBaseFieldDefinitions($entity_type);

    $fields['parent'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Parent note'))
      ->setDescription(t('The parent note if this is a reply.'))
      ->setSetting('target_type', 'moderation_note');

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The note author.'))
      ->setSetting('target_type', 'user')
      ->setRequired(TRUE);

    $fields['entity_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Notated entity type'))
      ->setDescription(t('The entity type this note is related to.'))
      ->setSetting('max_length', EntityTypeInterface::ID_MAX_LENGTH)
      ->setRequired(TRUE);

    $fields['entity_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Notated entity ID'))
      ->setDescription(t('The entity id this note is related to.'))
      ->setRequired(TRUE);

    $fields['entity_field_name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Notated entity field name'))
      ->setDescription(t('The field name this note is related to.'))
      ->setRequired(TRUE);

    $fields['entity_langcode'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Notated entity langcode'))
      ->setDescription(t('The language this note is related to.'))
      ->setRequired(TRUE);

    $fields['entity_view_mode_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Notated entity view mode ID'))
      ->setDescription(t('The entity view mode this note is related to.'))
      ->setRequired(TRUE);

    $fields['quote'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Quote'))
      ->setDescription(t('The quote that was selected, if applicable.'))
      ->setSetting('max_length', 255)
      ->setRequired(TRUE);

    $fields['quote_offset'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Quote Offset'))
      ->setDescription(t('The offset from the field ID for this quote.'))
      ->setRequired(TRUE);

    $fields['text'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Text'))
      ->setDescription(t('The text of the note.'))
      ->setSetting('max_length', FieldStorageConfig::NAME_MAX_LENGTH)
      ->setRequired(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'))
      ->setRequired(TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the note was last edited.'))
      ->setRequired(TRUE);

    $fields['assignee'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Assignee'))
      ->setDescription(t('The note assignee.'))
      ->setSetting('target_type', 'user');

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function hasParent() {
    return (bool) $this->get('parent')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function getParent() {
    if ($this->hasParent()) {
      return $this->get('parent')->entity;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getChildren() {
    $ids = \Drupal::entityQuery('moderation_note')
      ->accessCheck(FALSE)
      ->condition('parent', $this->id())
      ->sort('created')
      ->execute();
    return $this->entityTypeManager()->getStorage('moderation_note')->loadMultiple($ids);
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('uid')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->getEntityKey('uid');
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('uid', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('uid', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getModeratedEntity() {
    $entity_type = $this->getModeratedEntityTypeId();
    $entity_id = $this->getModeratedEntityId();
    $storage = $this->entityTypeManager()->getStorage($entity_type);
    return $storage->load($entity_id);
  }

  /**
   * {@inheritdoc}
   */
  public function setModeratedEntity(EntityInterface $entity) {
    if (!$entity->isNew()) {
      $this->set('entity_type', $entity->getEntityTypeId());
      $this->set('entity_id', $entity->id());
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getModeratedEntityTypeId() {
    return $this->get('entity_type')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getModeratedEntityId() {
    return $this->get('entity_id')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setModeratedEntityById($entity_type_id, $entity_id) {
    if ($storage = $this->entityTypeManager()->getStorage($entity_type_id)) {
      if ($storage->load($entity_id)) {
        $this->set('entity_type', $entity_type_id);
        $this->set('entity_id', $entity_id);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityFieldName() {
    return $this->get('entity_field_name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityFieldName($field_name) {
    $this->set('entity_field_name', $field_name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityLanguage() {
    return $this->get('entity_langcode')->getString();
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityLanguage($langcode) {
    $this->set('entity_langcode', $langcode);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityViewModeId() {
    return $this->get('entity_view_mode_id')->getString();
  }

  /**
   * {@inheritdoc}
   */
  public function setEntityViewModeId($view_mode_id) {
    $this->set('entity_view_mode_id', $view_mode_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getQuote() {
    return $this->get('quote')->getString();
  }

  /**
   * {@inheritdoc}
   */
  public function setQuote($quote) {
    $this->set('quote', $quote);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getQuoteOffset() {
    return $this->get('quote_offset')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setQuoteOffset($offset) {
    if (is_int($offset)) {
      $this->set('quote_offset', $offset);
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getText() {
    return $this->get('text')->getString();
  }

  /**
   * {@inheritdoc}
   */
  public function setText($text) {
    $this->set('text', $text);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getAssignee() {
    return $this->get('assignee')->entity ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function setAssignee(UserInterface $user) {
    $this->set('assignee', $user);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTagsToInvalidate() {
    $tags = parent::getCacheTagsToInvalidate();
    if (!empty($this->getModeratedEntityTypeId())) {
      $tags[] = implode(':', [
        'moderation_note',
        $this->getModeratedEntityTypeId(),
        $this->getModeratedEntityId(),
        $this->getEntityFieldName(),
        $this->getEntityLanguage(),
      ]);
    }
    return $tags;
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    parent::postSave($storage, $update);

    if (!$this->hasParent()) {
      $assignee = $this->get('assignee')->target_id;
      $type = $this->getModeratedEntityTypeId();
      $id = $this->getModeratedEntityId();

      Cache::invalidateTags([
        "moderation_note:user:$assignee",
        "moderation_note:$type:$id",
      ]);
    }

  }

  /**
   * {@inheritdoc}
   */
  public function toUrl($rel = 'canonical', array $options = []) {
    $entity = $this->getModeratedEntity();
    $options['query']['open-moderation-note'] = $this->id();
    $url = $entity->toUrl('canonical', $options);

    return $url;
  }

}
