<?php

namespace Drupal\node_revision_generate;

/**
 * Methods for generate revisions in a batch.
 */
class NodeRevisionGenerateBatch {

  /**
   * Implements callback_batch_operation().
   *
   * Generate revision for the node with $nid.
   *
   * @param int $nid
   *   The node ID.
   * @param int $revision_date
   *   Date to create the revision for the $nid node.
   * @param mixed $context
   *   An array of contextual key/values.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public static function generateRevisions(int $nid, int $revision_date, &$context): void {
    if (empty($context['results'])) {
      $context['results']['revisions'] = 0;
    }
    // Load the node to generate revision.
    /** @var \Drupal\node\NodeInterface $node_entity */
    $node_entity = \Drupal::entityTypeManager()->getStorage('node')->load($nid);

    // Generate revision.
    $node_entity->setNewRevision();
    $node_entity->setRevisionCreationTime($revision_date);
    $node_entity->setRevisionLogMessage(t('Revision generated by Node Revision Generate module.'));
    $node_entity->setRevisionUserId(\Drupal::currentUser()->id());
    // Calling this function let show revision in the user interface.
    $node_entity->setRevisionTranslationAffected(TRUE);
    $node_entity->save();

    // Count the number of revisions.
    $context['results']['revisions']++;
  }

  /**
   * Actions on finishing the batch.
   *
   * @param bool $success
   *   The flag to identify if batch has successfully run or not.
   * @param array $results
   *   The results from running context.
   * @param array $operations
   *   The array of operations remained unprocessed.
   */
  public static function finish(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();
    $logger = \Drupal::logger('node_revision_generate');

    if ($success) {
      $logger->notice(t('@total revisions generated for nodes', ['@total' => $results['revisions']]));

      $success_message = \Drupal::translation()->formatPlural(
        $results['revisions'],
        'One revision has been created for the selected content types.',
        'A total of @count revisions were created for the selected content types.',
        ['@total' => $results['revisions']]
      );
      $messenger->addMessage($success_message);
      $messenger->addWarning(t('Revisions were generated up to the current date, no revisions were generated with a date in the future. So, depending on this maybe we will not generate the number of revisions you expect.'));
    }
    else {
      // An error occurred.
      // $operations contains the operations that remained unprocessed.
      $error_operation = reset($operations);
      $message = t('An error occurred while processing %error_operation with arguments: @arguments', [
        '%error_operation' => $error_operation[0],
        '@arguments' => print_r($error_operation[1], TRUE),
      ]);
      $logger->error($message);
      $messenger->addError($message);
    }
  }

}
