<?php

namespace Drupal\Tests\scheduled_publish\Kernel;

use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\scheduled_publish\Service\ScheduledPublishCron;
use Drupal\Tests\field\Kernel\FieldKernelTestBase;
use Drupal\workflows\Entity\Workflow;

/**
 * Test the scheduled publish formatter.
 *
 * @package Drupal\Tests\scheduled_publish\Kernel
 * @group scheduled_publish
 */
class FormatterTest extends FieldKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'taxonomy',
    'scheduled_publish',
    'content_moderation',
    'workflows',
    'datetime',
    'user',
    'language',
    'config_translation',
    'locale',
  ];

  /**
   * The scheduled update service.
   *
   * @var \Drupal\scheduled_publish\Service\ScheduledPublishCron
   */
  private ScheduledPublishCron $scheduledUpdateService;

  /**
   * Language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  private LanguageManagerInterface $languageManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->setInstallProfile('standard');
    $this->installConfig([
      'field',
      'system',
      'content_moderation',
      'scheduled_publish',
      'language',
      'locale',
    ]);

    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installSchema('node', 'node_access');
    $this->installEntitySchema('user');
    $this->installEntitySchema('content_moderation_state');
    $this->installConfig('content_moderation');
    $this->installSchema('locale', [
      'locales_location',
      'locales_source',
      'locales_target',
    ]);

    $this->scheduledUpdateService = \Drupal::service('scheduled_publish.update');
    $this->createNodeType();

    $this->languageManager = \Drupal::languageManager();

    // Add France translation.
    ConfigurableLanguage::createFromLangcode('fr')->save();
    $configLanguageOverride = $this->container->get('language_manager')->getLanguageConfigOverride('fr', 'workflows.workflow.editorial');

    $configLanguageOverride
      ->set('type_settings.states.draft.label', 'Brouillon')
      ->set('type_settings.states.published.label', 'Publié')
      ->save();
  }

  /**
   * Creates a page node type to test with, ensuring that it's moderated.
   */
  protected function createNodeType() {
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'field_scheduled_publish',
      'type' => 'scheduled_publish',
      'entity_type' => 'node',
    ]);

    $field_storage->save();

    $node_type = NodeType::create([
      'type' => 'page',
      'name' => 'page',
    ]);
    $node_type->save();

    FieldConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_scheduled_publish',
      'bundle' => 'page',
      'label' => 'Test field',
    ])->save();

    $workflow = Workflow::load('editorial');
    /** @var \Drupal\content_moderation\Plugin\WorkflowType\ContentModerationInterface $contentModeration */
    $contentModeration = $workflow->getTypePlugin();
    $contentModeration->addEntityTypeAndBundle('node', 'page');
    $workflow->save();
  }

  /**
   * Test the rendered output.
   *
   * @dataProvider workflowStateLabelDataProvider
   */
  public function testRenderOutput(string $langcode, string $expected_label):void {
    $this->languageManager->setConfigOverrideLanguage(
      $this->languageManager->getLanguage($langcode)
    );

    $display = EntityViewDisplay::create([
      'targetEntityType' => 'node',
      'bundle' => 'page',
      'mode' => 'default',
      'status' => TRUE,
    ]);

    $display_options = [
      'label' => 'above',
      'type' => 'scheduled_publish_generic_formatter',
      'settings' => [
        'date_format' => 'html_date',
        'text_pattern' => '%moderation_state% ------ %date%',
      ],
    ];
    $display->setComponent('field_scheduled_publish', $display_options);

    $display->save();
    $page = Node::create([
      'type' => 'page',
      'title' => 'A',
      'uid' => 1,
    ]);

    $page->moderation_state->value = 'draft';
    $page->set('field_scheduled_publish', [
      'moderation_state' => 'published',
      'value' => '2021-01-31T00:00:00',
    ]);
    $page->save();

    $nodeID = $page->id();

    $view_builder = \Drupal::entityTypeManager()->getViewBuilder('node');
    $storage = \Drupal::entityTypeManager()->getStorage('node');
    $node = $storage->load($nodeID);
    $build = $view_builder->view($node, 'default');

    if (method_exists(RendererInterface::class, 'renderPlain')) {
      // @phpstan-ignore-next-line as it is deprecated in D10.3 and removed from D12.
      $output = \Drupal::service('renderer')->renderPlain($build);
    }
    else {
      $output = \Drupal::service('renderer')->renderInIsolation($build);
    }

    self::assertTrue((bool) $nodeID);
    self::assertNotFalse(strpos($output, "<div>$expected_label ------ 2021-01-31</div>"));

    $this->scheduledUpdateService->doUpdate();

    Node::load($nodeID);
  }

  /**
   * Provides cases with expected workflow state labels in different languages.
   *
   * @return array
   *   A list of test cases.
   */
  public static function workflowStateLabelDataProvider() {
    return [
      [
        'langcode' => 'en',
        'expected_label' => 'Published',
      ],
      [
        'langcode' => 'fr',
        'expected_label' => 'Publié',
      ],
    ];
  }

}
