<?php

namespace Drupal\w3c_validator\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\w3c_validator\W3CProcessor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the operation form on report page.
 */
class W3CValidatorOperationForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The W3C process service.
   *
   * @var \Drupal\w3c_validator\W3CProcessor
   */
  protected $w3cProcessor;

  /**
   * Constructs a new W3CValidatorOperationForm.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\w3c_validator\W3CProcessor $w3c_processor
   *   The validation processor.
   */
  public function __construct(Connection $connection, W3CProcessor $w3c_processor) {
    $this->connection = $connection;
    $this->w3cProcessor = $w3c_processor;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('w3c.processor')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'w3c_validator_operation_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $validator_url = $this->w3cProcessor->getValidatorUrl();

    $warning = '';
    if ($this->isRateLimitService($validator_url)) {
      $warning = $this->t('This will revalidate all of the following pages. This operation may be long.');
      $warning .= '<br/><br/><i><b>' . $this->t('BEWARE : You are using the W3C HTML Validator at @path.', [
        '@path' => Link::fromTextAndUrl($validator_url, Url::fromUri($validator_url))
          ->toString(),
      ]) . '<br/>' . $this->t('Using an external online service may be consider spam and abuse of service. Therefore, performing this operation, you may get banned temporarily.') . '<br/>' .
        $this->t('Configure a local service on @configuration', [
          '@configuration' => Link::createFromRoute('W3C validator settings page', 'w3c_validator.settings')
            ->toString(),
        ]) . '</b></i>';
    }

    $form['advanced_operations'] = [
      '#type' => 'details',
      '#title' => $this->t('advanced operations'),
      '#description' => $warning,
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    // Load module settings.
    $module_settings = $this->configFactory()->get('w3c_validator.settings');

    // Use admin helper tool option settings.
    $form['advanced_operations']['use_token'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Validate as logged-in user.'),
      '#description' => $this->t('If enabled, pages will be validated as you can see it. Otherwise, as an anonymous user.'),
      '#default_value' => $module_settings->get('use_token'),
    ];

    // Include admin pages.
    $form['advanced_operations']['admin_pages'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include admin pages.'),
      '#description' => $this->t('If enabled, admin pages will be included in validation.'),
      '#default_value' => $module_settings->get('admin_pages'),
    ];

    $form['advanced_operations']['w3c_validator_revalidate_all'] = [
      '#type' => 'submit',
      '#value' => 'Re-validate all pages',
      '#prefix' => '<br/>',
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    // Save global configurations.
    $this->configFactory->getEditable('w3c_validator.settings')
      ->set('use_token', $values['use_token'])
      ->set('admin_pages', $values['admin_pages'])
      ->save();

    $form_state->setRedirect('w3c_validator.confirm');
  }

  /**
   * Checks if service belong to a known rate limited service.
   *
   * @param string $service
   *   The service name.
   *
   * @return bool
   *   true/false if the service is know to have a rate limit.
   */
  protected function isRateLimitService($service) {
    $services = [
      'validator.nu',
      'validator.w3.org',
    ];
    return in_array(parse_url($service, PHP_URL_HOST), $services);
  }

}
