<?php

namespace Drupal\w3c_validator\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\w3c_validator\W3CProcessor;
use HtmlValidator\Validator;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration settings page for W3C Validator page.
 *
 * @internal
 */
class W3cValidatorSettingsForm extends ConfigFormBase {

  /**
   * The W3C validation processor.
   *
   * @var \Drupal\w3c_validator\W3CProcessor
   */
  protected $w3cProcessor;

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\w3c_validator\W3CProcessor $w3c_processor
   *   The w3c validation processor.
   */
  public function __construct(ConfigFactoryInterface $config_factory, W3CProcessor $w3c_processor) {
    parent::__construct($config_factory);
    $this->w3cProcessor = $w3c_processor;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('w3c.processor')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'w3c_validator_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['w3c_validator.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    $form['w3c_validator'] = [
      '#markup'  =>
      $this->t('This module uses the official W3c Validator scripts. We strongly recommand to install the official w3c validator on your own server and provide the url here? You can also use the official online w3c validator service.') . '<br />' .
      $this->t('For instructions on how to install an instance of the w3c validator scripts you have instructions at <a href="http://validator.w3.org/docs/install.html">http://validator.w3.org/docs/install.html</a>.') . '<br />' .
      $this->t('Debian based distributions comes with a package called <em>w3c_markup_validator</em> which provides a ready to use install of the validator.') . '<br />' .
      $this->t('Using the official W3C validator endpoint url for high volume of validations could be considered abuse of service.'),
    ];

    $form['validator_url'] = [
      '#type' => 'url',
      '#title' => $this->t('W3C Validator API endpoint URI'),
      '#description' => $this->t('URL of the validator script where API call will be targeted.') . '<br/>' .
      $this->t('You can use the official service at @w3cvalidator althought its not recommended.', [
        '@w3cvalidator' => Link::fromTextAndUrl('http://validator.w3.org/check', Url::fromUri('http://validator.w3.org/check'))->toString()
      ]) . '<br/>' .
      $this->t('Leave empty to use the default installed validator path: @validator.', [
        '@validator' => Link::fromTextAndUrl(Validator::DEFAULT_VALIDATOR_URL, Url::fromUri(Validator::DEFAULT_VALIDATOR_URL))->toString()
      ]),
      '#default_value' => $this->w3cProcessor->getValidatorUrl(),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $this->config('w3c_validator.settings')
      ->set('validator_url', $form_state->getValue('validator_url'))
      ->save();
  }

}
