<?php

namespace Drupal\goutte\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\file\Entity\File;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\goutte\BatchProcessor;

/**
 * Class GoutteForm extends FormBase.
 */
class GoutteForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'goutte_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['csv'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('CSV'),
      '#upload_location' => 'public://csv/',
      "#upload_validators"  => ['file_validate_extensions' => ['csv']],
      '#description' => $this->t('File including a list of web addresses to migrate.'),
    ];
    $form['url'] = [
      '#type' => 'url',
      '#title' => $this->t('Web Address'),
      '#description' => $this->t('A single web address to migrate.'),
    ];

    $form['title_filter'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title Filter'),
      '#description' => $this->t('The filter to be used by goutte to find the title.'),
      '#maxlength' => 255,
      '#size' => 64,
    ];

    $form['content_filter'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Content Filter'),
      '#description' => $this->t('The filter to be used by goutte to find the content.'),
      '#maxlength' => 255,
      '#size' => 64,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    // In this variable you will have file entity.
    $file = $form_state->getValue('csv');

    $file = File::load($file[0]);
    $file->setPermanent();
    $file->save();

    // You can use any sort of function to process your data.
    // The goal is to get each 'row' of data into an array.
    // If you need to work on how data is extracted, process it here.
    $rows = $this->csvtoarray(',', $file->getFileUri());

    $batch_processor = new BatchProcessor();

    $batch_builder = (new BatchBuilder())
      ->setTitle(t('Goutte content import'))
      ->setFinishCallback([$batch_processor, 'batchImportContentCallback'])
      ->setInitMessage(t('Commencing content import.'))
      ->setProgressMessage(t('Processed @current out of @total. Estimated time: @estimate.'))
      ->setErrorMessage(t('The migration process has encountered an error.'));

    foreach ($rows as $row) {
      $batch_builder->addOperation([$batch_processor, 'batchImportContent'], [$row]);
    }

    batch_set($batch_builder->toArray());

  }

  /**
   * Parse the CSV file.
   *
   * Borrowing this code from:
   * https://www.fuseiq.com/blog/import-drupal-8-content-csv-file.
   */
  public function csvtoarray($delimiter, $filename = '') {
    if (!file_exists($filename) || !is_readable($filename)) {
      return FALSE;
    }
    $header = NULL;
    $data = [];
    if (($handle = fopen($filename, 'r')) !== FALSE) {
      while (($row = fgetcsv($handle, 1000, $delimiter)) !== FALSE) {
        if (!$header) {
          $header = $row;
        }
        else {
          $data[] = array_combine($header, $row);
        }
      }
      fclose($handle);
    }
    return $data;
  }

}
