<?php

namespace Drupal\goutte;

use Drupal\node\Entity\Node;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\file\Entity\File;
use Drupal\taxonomy\Entity\Term;
use Drupal\redirect\Entity\Redirect;
use Drupal\Core\Url;
use Drupal\menu_link_content\Entity\MenuLinkContent;

/**
 * NodeBuilder class imports NodeBuilder.
 */
class NodeBuilder {

  /**
   * Short description for $node.
   *
   * @var Drupal\node\Entity\Node
   */
  protected $node;

  /**
   * Short description for $normalizer.
   *
   * @var Drupal\goutte\NodeNormalizer
   */
  protected $normalizer;

  /**
   * Short description for $title.
   *
   * @var string
   */
  protected $title;

  /**
   * Short description for $url.
   *
   * @var string
   */
  protected $url;

  /**
   * Short description for $type.
   *
   * @var string
   */
  protected $type;

  /**
   * Short description for $comments.
   *
   * @var array
   */
  protected $comments;

  /**
   * Short description for $featuredImage.
   *
   * @var string
   */
  protected $featuredImage;

  /**
   * Short description for $internalLinks.
   *
   * @var array
   */
  protected $internalLinks;

  /**
   * Short description for $MenuId.
   *
   * @var string
   */
  protected $menuId;

  /**
   * Short description for $KnowHierarchy.
   *
   * @var string
   */
  protected $knowHierarchy;

  /**
   * Short description for $WebStructure.
   *
   * @var string
   */
  protected $webStructure;

  /**
   * Get variable.
   */
  public function getInternalLinks() {
    return $this->internalLinks;
  }

  /**
   * Set variable.
   */
  public function setInternalLinks($value) {
    $this->internalLinks[] = $value;
  }

  /**
   * Get variable.
   */
  public function getType() {
    return $this->type;
  }

  /**
   * Set variable.
   */
  public function setType($value) {
    $this->type = $value;
  }

  /**
   * Get variable.
   */
  public function getUrl() {
    return $this->url;
  }

  /**
   * Set variable.
   */
  public function setUrl($value) {
    $this->url = $value;
  }

  /**
   * Get variable.
   */
  public function getMenuId() {
    return $this->menuId;
  }

  /**
   * Set variable.
   */
  public function setMenuId($value) {
    $this->MenuId = $value;
  }

  /**
   * Get variable.
   */
  public function getKnowHierarchy() {
    return $this->knowHierarchy;
  }

  /**
   * Set variable.
   */
  public function setKnowHierarchy($value) {
    $this->KnowHierarchy = $value;
  }

  /**
   * Get variable.
   */
  public function getWebStructure() {
    return $this->webStructure;
  }

  /**
   * Set variable.
   */
  public function setWebStructure($value) {
    $this->WebStructure = $value;
  }

  /**
   * Get variable.
   */
  public function getTitle() {
    return $this->title;
  }

  /**
   * Set variable.
   */
  public function setTitle($value) {
    $this->title = $value;
  }

  /**
   * Get variable.
   */
  public function getId() {
    return $this->node->id();
  }

  /**
   * Get variable.
   */
  public function getComments() {
    return $this->comments;
  }

  /**
   * Set variable.
   */
  public function setComments($value) {
    $this->comments = $value;
  }

  /**
   * Get variable.
   */
  public function getImage() {
    return $this->featured_image;
  }

  /**
   * Set variable.
   */
  public function setImage($value) {
    $this->featured_image = $value;
  }

  /**
   * Get variable.
   */
  public function getLegacyUrl() {
    return $this->get('field_legacy_url')->getValue();
  }

  /**
   * Class constructor.
   */
  public function __construct() {
    $this->normalizer = new NodeNormalizer();
  }

  /**
   * Set variable.
   */
  public function setUp($type) {
    $this->node = Node::create(['type' => $type]);
    $this->setType($type);
  }

  /**
   * Set field.
   */
  public function setTitleField($field, $content) {
    $value = $this->normalizer->normalizeTitle($content);
    $this->setTitle($value);
    $this->node->set($field, $value);
  }

  /**
   * Set field.
   */
  public function setLegacyField($field, $content) {
    $value = $this->normalizer->normalizeLegacy($content);
    $this->node->set($field, $value);
  }

  /**
   * Set field.
   */
  public function setBodyField($field, $content) {
    $value = [
      'summary' => '',
      'value' => $this->normalizer->normalizeTextarea($content),
      'format' => 'rich_text',
    ];
    $this->node->set($field, $value);
  }

  /**
   * Set field.
   */
  public function setParagraphField($field, $content) {
    $paragraph = Paragraph::create([
      'type' => 'text',
      'field_text' => [
        'value' => $this->normalizer->normalizeTextarea($content),
        'format' => 'rich_text',
      ],
    ]);
    $paragraph->save();
    $value = [
      'target_id' => $paragraph->id(),
      'target_revision_id' => $paragraph->getRevisionId(),
    ];
    $this->node->set($field, $value);
  }

  /**
   * Set field.
   */
  public function setSummaryField($field, $content) {
    $value = $this->normalizer->normalizeSummary($content);
    $this->node->set($field, $value);
  }

  /**
   * Set field.
   */
  public function setDateField($field, $content, $format) {
    if ($format == 'UNIX') {
      $value = $this->normalizer->normalizeDate($content);
    }
    else {
      $value = $this->normalizer->normalizeDateField($content);
    }
    $this->node->set($field, $value);
  }

  /**
   * Helper function to create a taxonomy term programmatically.
   */
  public function setTaxonomyField($field, $vocabulary, $terms) {
    if (!empty($terms)) {
      foreach ($terms as $term_name) {
        if (!empty($term_name)) {
          if ($tid = $this->getTidByName($term_name, $vocabulary)) {
            $node_terms[] = $tid;
          }
          else {
            $term = Term::create([
              'name' => $this->normalizer->normalizeTerm($term_name),
              'vid'  => $vocabulary,
            ])->save();
            if ($tid = $this->getTidByName($term_name, $vocabulary)) {
              $node_terms[] = $tid;
            }
          }
        }
      }
      if (isset($node_terms)) {
        $this->node->set($field, $node_terms);
      }
    }
  }

  /**
   * Load term by name.
   */
  protected function getTidByName($name = NULL, $vocabulary = NULL) {
    $properties = [];
    if (!empty($name)) {
      $properties['name'] = $name;
    }
    if (!empty($vocabulary)) {
      $properties['vid'] = $vocabulary;
    }
    $terms = \Drupal::entityManager()->getStorage('taxonomy_term')->loadByProperties($properties);
    $term = reset($terms);
    return !empty($term) ? $term->id() : 0;
  }

  /**
   * Set field.
   */
  public function setImageField($field) {
    if (!empty($this->getImage())) {
      $value = [
        'target_id' => $this->getImage(),
        'alt' => $this->getTitle(),
        'title' => $this->getTitle(),
      ];
      $this->node->set($field, $value);
    }
  }

  /**
   * Helper to upload file.
   */
  public function uploadFile($files, $file_type) {

    if (!empty($files)) {
      foreach ($files as $file) {

        // Determine the absolute path, if does not exist.
        if (strrpos($file, '.ca') !== FALSE) {

          $domain = substr($file, 0, strrpos($file, '.ca') + 2);
          $file_path = str_replace($domain, '', substr($file, 0, strrpos($file, '/', -1) + 1));

        }
        elseif (strpos($file, '.com') !== FALSE) {

          $domain = substr($file, 0, strrpos($file, '.com') + 3);
          $file_path = str_replace($domain, '', substr($file, 0, strrpos($file, '/', -1) + 1));

        }
        elseif (strpos($file, '../') !== FALSE) {
          $domain = '';

          // http://www.yrdsb.ca/aboutus/departments/planning/pages/secondary-boundaries.aspx
          $arr = explode('/', $this->getUrl());
          // Removes filename.ext of the URL.
          // example: secondary-boundaries.aspx.
          array_pop($arr);

          // Finds number of relative folder structure.
          $count = substr_count($file, '../');
          for ($i = 0; $i < $count; $i++) {
            array_pop($arr);
          }

          // Rebuilds the file path ../Documents/Boundary/416Boundary.pdf.
          $url = implode('/', $arr);
          $file = str_replace('../', '', $file);
          $file_path = $url . '/' . $file;
          $file_path = substr($file_path, 0, strrpos($file_path, '/', -1) + 1);
        }
        else {

          $domain = 'http://www.yrdsb.ca';
          $file_path = substr($file, 0, strrpos($file, '/', -1) + 1);

        }

        $file_name = substr($file, strrpos($file, '/', -1) + 1);

        if ($file_name) {
          $path = 'public://migrate/files/' . urldecode(basename($file_name));
          $saved_file = file_get_contents($domain . $file_path . html_entity_decode($file_name));

          // If the image can be retrieved, and has more than 0 bytes.
          if (strlen($saved_file) > 0 && $saved_file !== FALSE) {
            $insert = file_put_contents($path, $saved_file);

            // Create the file locally.
            if ($insert) {
              $local_file = File::create([
                'uid' => 1,
                'filename' => urldecode(basename($file_name)),
                'uri' => 'public://migrate/files/' . urldecode(basename($file_name)),
                'status' => 1,
              ]);
              $local_file->save();
            }

            // Create a redirect for the legacy file.
            $source = $file_path . urldecode(basename($file_name));
            $destination = '/sites/default/files/migrate/files/' . urldecode(basename($file_name));
            $this->setRedirect($source, 'internal:' . $destination);

            // Store this for future processing.
            $this->setInternalLinks([$file, $destination]);

            // Set the featured image.
            if ($file_type == "image") {
              if (empty($this->getImage())) {
                $this->setImage($local_file->id());
              }
            }
          }
        }
      }
    }
  }

  /**
   * Helper to set redirect.
   */
  public function setRedirect($source, $destination) {

    $redirects = \Drupal::service('redirect.repository')->findBySourcePath($source);
    // Create the redirect, if one does not exist.
    if (empty($redirects)) {
      $redirect = Redirect::create([
        'redirect_source' => $source,
        'redirect_redirect' => $destination,
        'language' => 'und',
        'status_code' => '301',
      ]);
      $redirect->save();
    }
  }

  /**
   * Sets menu link after the node is saved.
   */
  public function setMenuLink($know_hierarchy, $web_structure) {
    $nid = 0;

    // Check nid of url alias if know_hierarchy is not null.
    if (isset($know_hierarchy) && !empty($know_hierarchy) > 0) {
      $path = \Drupal::service('path.alias_manager')->getPathByAlias('/' . $know_hierarchy);

      \Drupal::logger('Goutte KnowHierarchy')->notice($path);
      if (preg_match('/node\/(\d+)/', $path, $matches)) {
        $nid = $matches[1];
      }
    }

    // Check nid of legacy url if web_structure is not null.
    if (isset($web_structure) && !empty($web_structure)) {
      $suffix = 'Pages/default.aspx';
      \Drupal::logger('Goutte WebStructure')->notice('/' . $web_structure . '/' . $suffix);
      $query = \Drupal::entityQuery('node')
        ->condition('field_legacy_url', '/' . $web_structure . '/' . $suffix);
      $nids = $query->execute();
      \Drupal::logger('Goutte WebStructure')->notice(var_export($nids, TRUE));
      $nid = end($nids);
    }

    \Drupal::logger('Goutte Menu Link NID')->notice($nid);

    if (isset($nid) && $nid > 0) {
      MenuLinkContent::create([
        'title' => $this->getTitle(),
        'link' => ['uri' => 'entity:node/' . $this->getId()],
        'menu_name' => $this->getMenuId(),
        'status' => TRUE,
        // 'parent' => $parent,
        'weight' => 0,
      ])->save();
    }
  }

  /**
   * Unset features.
   */
  public function tearDown($legacy_url) {
    $query = \Drupal::entityQuery('node')
      ->condition('field_legacy_url', '/' . $legacy_url);
    $nids = $query->execute();

    if (isset($nids) && count($nids) > 0) {
      // Do something with already processed url.
      foreach ($nids as $key => $nid) {
        $node = Node::load($nid);
        $bodytmp = $this->node->body->getValue();
        $bodytmp = (isset($bodytmp[0]['value'])) ? $bodytmp[0]['value'] : '';
        if ($node->body->value != $bodytmp) {
          $node->setNewRevision(FALSE);
          $node->body->value = $bodytmp;
          $node->save();
        }
      }
    }
    else {
      // Determine the relative source path.
      $source = $legacy_url;

      $this->node->revision_log = 'Content migrated from http://www.yrdsb.ca/' . $source;
      $this->node->set('moderation_state', "published");
      $this->node->save();

      // Create the redirect, if one does not exist.
      $destination = 'internal:/node/' . $this->getId();
      $this->setRedirect($source, $destination);

      // Set the menu link.
      // $this->setMenuLink($this->getKnowHierarchy(), $this->getWebStructure())
      // Set the new URL.
      $options = ['absolute' => FALSE];
      $this->setUrl(Url::fromRoute('entity.node.canonical', ['node' => $this->getId()], $options)->toString());
    }
  }

}
