<?php

namespace Drupal\york_bww_integration\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\Node;
use Drupal\file\Entity\File;

/**
 * Provides a form for changing Consumer API Keys and Access Tokens.
 */
class YorkBWWIntegrationForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'york_bww_integration';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['file_xml_upload'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('XML File Upload'),
      '#field_name' => 'file_xml_upload',
      '#description' => t('XML format only.'),
      '#upload_location' => 'public://bww_files',
      '#upload_validators' => [
        'file_validate_extensions' => ['xml'],
      ],
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Import BWW data'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    $form_file = $form_state->getValue('file_xml_upload', 0);
    if (isset($form_file[0]) && !empty($form_file[0])) {
      $file = File::load($form_file[0]);
      $new_filename = "BWW.xml";
      $stream_wrapper = \Drupal::service('file_system')->uriScheme($file->getFileUri());
      $new_filename_uri = "{$stream_wrapper}://bww_files/{$new_filename}";

      if (file_exists('sites/default/files/bww_files/BWW.xml') && $file->getFilename() != 'BWW.xml') {
        unlink('sites/default/files/bww_files/BWW.xml');
      }

      rename('sites/default/files/bww_files/' . $file->getFilename(), 'sites/default/files/bww_files/BWW.xml');

      $file->setPermanent();
      $file->save();
    }

    $yorkXmlFile = simplexml_load_file('public://bww_files/BWW.xml');

    $drupalCurrentNodes = [];
    $drupalCurrentXMLIds = [];
    $yorkParsed = [];
    $yorkParsedBWWIds = [];
    $yorkUpdate = [];
    $yorkAdd = [];
    $yorkDelete = [];

    if (count($yorkXmlFile->entry) > 0) {
      // All Data from Drupal.
      $nids = \Drupal::entityQuery('node')->condition('type', 'policies_procedures')->execute();
      $nodes = Node::loadMultiple($nids);

      foreach ($nodes as $node) {
        $pp_number_node = $node->get('field_bww_integration_tracking_n')->getValue()[0]['value'];
        $xmlId = $node->get('field_xmlid')->getValue()[0]['value'];
        $pp_type_number = $this->getTidByName($node->get('field_type')->getValue()[0]['target_id'], 'policies_and_procedures');
        $field_last_reviewed = (isset($node->get('field_last_reviewed')->getValue()[0]['value'])) ? $node->get('field_last_reviewed')->getValue()[0]['value'] : '';

        $currentDrupalNode = [
          'xmlId' => $xmlId,
          'node_id' => $node->id(),
          'type' => 'policies_procedures',
          'title' => $node->get('title')->getValue()[0]['value'],
          'field_bww_integration_tracking_n' => $pp_number_node,
          'field_last_reviewed' => $field_last_reviewed,
          'field_type' => $pp_type_number,
        ];
        array_push($drupalCurrentNodes, $currentDrupalNode);
        $drupalCurrentXMLIds[] = $xmlId;
      }

      // All Data from XML.
      foreach ($yorkXmlFile->entry as $en) {
        $title = $en->children('m', TRUE)->children('d', TRUE)->Filename;
        $pp_number = $en->children('m', TRUE)->children('d', TRUE)->PPNumber;
        $pp_last_reviewed = (string) $en->children('m', TRUE)->children('d', TRUE)->PPLastReviewed;
        $pp_last_reviewed = substr($pp_last_reviewed, -4);
        $yorkFileToDownload = $en->content->attributes()->src;
        $yorkFileName = $en->children('m', TRUE)->children('d', TRUE)->Name;

        if (!$pp_last_reviewed == NULL) {
          $time = strtotime($pp_last_reviewed);
          $pp_last_reviewed = date($pp_last_reviewed . '-m-d', $time);
        }

        $title = ($title == '' ? '-' : $title);
        $pp_number = ($pp_number == '' ? '-' : $pp_number);
        $pp_type = $en->children('m', TRUE)->children('d', TRUE)->CategoryValue;
        $xmlId = $en->children('m', TRUE)->children('d', TRUE)->Id;
        $pp_type_number = $this->getTidByName($pp_type, 'policies_and_procedures');

        $currentXmlNode = [
          'xmlId' => (string) $xmlId,
          'type' => 'policies_procedures',
          'title' => (string) $title,
          'field_bww_integration_tracking_n' => (string) $pp_number,
          'field_last_reviewed' => $pp_last_reviewed,
          'field_type' => (string) $pp_type_number,
          'file_name' => $yorkFileName,
          'file_url' => $yorkFileToDownload,
        ];

        array_push($yorkParsed, $currentXmlNode);
        $yorkParsedBWWIds[] = (string) $xmlId;
      }

      $yorkUpdate = array_intersect($yorkParsedBWWIds, $drupalCurrentXMLIds);
      $yorkAdd = array_diff($yorkParsedBWWIds, $drupalCurrentXMLIds);
      $yorkDelete = array_diff($drupalCurrentXMLIds, $yorkParsedBWWIds);

      // ADD entries.
      $addMessage = $this->addNode($yorkAdd, $yorkParsed);
      // UPDATE entries.
      $updateMessage = $this->updateNode($yorkUpdate, $yorkParsed, $drupalCurrentNodes);
      // DELETE entries.
      $deleteMessage = $this->deleteNode($yorkDelete, $drupalCurrentNodes);

      $this->messenger()->addStatus($addMessage);
      $this->messenger()->addStatus($updateMessage);
      $this->messenger()->addStatus($deleteMessage);
    }

  }

  /**
   * {@inheritdoc}
   */
  public function updateNode($yorkUpdate, $yorkParsed, $drupalCurrentNodes) {
    echo "<p>Update Node is activated</p>";
    $updated = 0;
    foreach ($yorkUpdate as $yorkKey => $val) {
      $nodeToUpdateIndex = array_search($val, array_column($drupalCurrentNodes, 'xmlId'));
      $nodeToUpdateID = $drupalCurrentNodes[$nodeToUpdateIndex]['node_id'];
      $yorkNode = $yorkParsed[$yorkKey];

      $node = Node::load($nodeToUpdateID);
      if (!$this->checkIfUpdateNeeded($node, $yorkNode)) {
        continue;
      }
      $node->field_type = $yorkNode['type'];
      $node->title = $yorkNode['title'];
      $node->field_bww_integration_tracking_n = $yorkNode['field_bww_integration_tracking_n'];
      $node->field_type = $yorkNode['field_type'];
      $node->moderation_state = 'published';
      $node->field_last_reviewed = $yorkNode['field_last_reviewed'];
      $this->uploadDownloadFile($node, $yorkNode['file_name'], $yorkNode['file_url']);

      // Save to update node.
      $updateNode = $node->save();
      if ($updateNode == 2) {
        $updated++;
      }
    }
    return $updated . " documents have been Updated.";
  }

  /**
   * {@inheritdoc}
   */
  public function addNode($yorkAdd, $yorkParsed) {
    echo "<p>Add Node is activated</p>";
    $saved = 0;
    foreach ($yorkAdd as $yorkKey => $val) {

      $keyFrom = array_search($val, array_column($yorkParsed, 'xmlId'));
      $yorkNode = $yorkParsed[$keyFrom];

      $node = Node::create([
        'type'        => 'policies_procedures',
        'title'       => $yorkNode['title'],
        'field_bww_integration_tracking_n' => $yorkNode['field_bww_integration_tracking_n'],
        'field_last_reviewed' => $yorkNode['field_last_reviewed'],
        'field_xmlid' => $yorkNode['xmlId'],
        'field_type' => $yorkNode['field_type'],
        'moderation_state' => 'published',
      ]);
      $this->uploadDownloadFile($node, $yorkNode['file_name'], $yorkNode['file_url']);
      $saveNode = $node->save();
      if ($saveNode == 1) {
        $saved++;
      }
    }
    return $saved . " documents have been Added.";
  }

  /**
   * {@inheritdoc}
   */
  public function deleteNode($yorkDelete, $drupalCurrentNodes) {
    echo "<p>Delete Node is activated</p>";
    $deleted = 0;
    foreach ($yorkDelete as $yorkKey => $val) {
      $nodeToDeleteIndex = array_search($val, array_column($drupalCurrentNodes, 'xmlId'));
      $nodeToDeleteId = $drupalCurrentNodes[$nodeToDeleteIndex]['node_id'];
      $node = Node::load($nodeToDeleteId);
      if (count($node) > 0) {
        $deleteNode = $node->delete();
        if ($deleteNode == 0) {
          $deleted++;
        }
      }
    }
    return $deleted . " documents have been deleted.";
  }

  /**
   * Load term by name.
   */
  protected function getTidByName($name = NULL, $vocabulary = NULL) {
    $properties = [];
    if (!empty($name)) {
      $properties['name'] = $name;
    }
    if (!empty($vocabulary)) {
      $properties['vid'] = $vocabulary;
    }
    $terms = \Drupal::entityManager()->getStorage('taxonomy_term')->loadByProperties($properties);
    $term = reset($terms);
    return !empty($term) ? $term->id() : 0;
  }

  /**
   * This function takes a node and file path.
   *
   * Uploads the file to the node (does not save).
   */
  private function uploadDownloadFile(&$node, $fileName, $yorkFileToDownload) {

    // Call and download file here.
    $drupalFilePathName = 'sites/default/files/bww_files/' . $fileName;
    $url = $yorkFileToDownload;
    $curlCh = curl_init();
    curl_setopt($curlCh, CURLOPT_URL, $url);
    curl_setopt($curlCh, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($curlCh, CURLOPT_SSLVERSION, 3);
    $curlData = curl_exec($curlCh);
    curl_close($curlCh);
    $downloadPath = $drupalFilePathName;
    $file = fopen($downloadPath, "w+");
    fwrite($file, $curlData);
    fclose($file);

    // Create file object from a locally copied file.
    $uri = \Drupal::service('file_system')->copy($drupalFilePathName, 'public://', FILE_EXISTS_REPLACE);
    $file = File::Create([
      'uri' => $uri,
      'status' => 1,
    ]);
    $file->save();
    // Attach file to node.
    $node->field_pdf->setValue([
      'target_id' => $file->id(),
    ]);
  }

  /**
   * Function to check if Update is needed.
   */
  private function checkIfUpdateNeeded($node, $xmlNode) {
    $needsUpdating = FALSE;
    $node_title = $node->getTitle();
    $node_pp_number = $node->get('field_bww_integration_tracking_n')->getValue()[0]['value'];
    $node_pp_last_reviewed = (isset($node->get('field_last_reviewed')->getValue()[0]['value'])) ? $node->get('field_last_reviewed')->getValue()[0]['value'] : '';
    $formatted_xml_date = substr($xmlNode['field_last_reviewed'], 0, 4);
    $formatted_node_date = substr($node_pp_last_reviewed, 0, 4);
    if ($xmlNode['title'] != $node_title
        || $xmlNode['field_bww_integration_tracking_n'] != $node_pp_number
        || $formatted_xml_date != $formatted_node_date) {
      $needsUpdating = TRUE;

    }
    return $needsUpdating;
  }

}
