<?php

namespace Drupal\york_google_calendar_sync\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Class SettingsForm extends ConfigFormBase.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'york_google_calendar_sync.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('york_google_calendar_sync.settings');

    $form['gc_client_credentials'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Google Calendar Credentials'),
      '#description' => $this->t('The auto-generated credentials.json file'),
      '#upload_location' => 'private://oauth/google_calendar/',
      '#default_value' => $config->get('gc_client_credentials'),
      '#upload_validators' => [
        'file_validate_extensions' => ['json'],
      ],
    ];

    $description = '
    If you do not have Oauth credentials set up, you must create one by creating a project (or adding to an existing one), enabling the Google Calendar API, setting up Open Authentication and exporting your credentials.

    <h3>Create a project</h3>

    <ol>
      <li>Log in to <a title="Google Cloud Console" href="https://console.developers.google.com/" target="_blank">Google Cloud Console</a> (this should be done with an administrator account if possible)</li>
      <li>Create a <a title="New Project" href="https://console.cloud.google.com/projectcreate" target="_blank">New Project</a>, and press <strong>Create</strong>. It will take a few minutes for the project to complete the set up.</li>
    </ol>

    <h3>Enable the Google Calendar API</h3>
    <ol>
      <li>Enable the <a title="Google Calendar API" href="https://console.cloud.google.com/apis/library/calendar-json.googleapis.com" target="_blank">Google Calendar API</a> in your project.</li>
      <li>Create an Oauth Conse the <a title="Google OAuth Consent" href="https://console.cloud.google.com/apis/credentials/consent" target="_blank">Create an OAuth ID</a> in your project.</li>
    </ol>

    <h3>Setup OAuth</h3>
    <ol>
      <li>Visit the <a title="Google OAuth Consent" href="https://console.cloud.google.com/apis/credentials/consent" target="_blank">OAuth Consent</a> page</li>
      <li>Select <strong>Internal</strong>, press <strong>Create</strong></li>
      <li>Set an <strong>Application Name</strong> that users will see</li>
      <li>Set your site domain as an <strong>Authorized Domain</strong> (e.g. example.com, do not include subdomains, http:// or https://)
    </ol>

    <h3>Create your Credentials</h3>
    <ol>
      <li>Visit the <a title="Google Auth Credentials Client" href="https://console.cloud.google.com/apis/credentials/oauthclient" target="_blank">OAuth Credentials Client</a> page</li>
      <li>Select <strong>Web application</strong></li>
      <li>Add a name for your application <strong>Name</strong></li>
      <li>Set the <strong>Authorized redirect URI</strong> to your website domain, following this pattern: https://example.com/admin/config/york_google_calendar_sync/settings</li>
    </ol>

    <h3>Download your Credentials</h3>
    <ol>
      <li>Visit the <a title="Google Auth Credentials" href="https://console.cloud.google.com/apis/credentials" target="_blank">OAuth Credentials</a> page</li>
      <li>Select the <strong>Download Icon</strong> of the OAuth client listed under <em>OAuth 2.0 client IDs</em></li>
      <li>Use this credentials.json file above</li>
    </ol>';

    $form['description'] = [
      '#type' => 'details',
      '#title' => 'Setting up your Google API credentials',
      '#description' => $description,
      '#open' => FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $config = $this->config('york_google_calendar_sync.settings');

    // Saves credentials to DB.
    $this->saveSettings($form, $form_state);

    if (isset($_GET['code'])) {

      $client = $this->getClient($form, $form_state, $_GET['code']);

      $tokenPath = 'private://oauth/google_calendar/token.json';
      if (file_exists(dirname($tokenPath))) {

        $service = new \Google_Service_Calendar($client);

        // Print the next 10 events on the user's calendar.
        $calendarId = 'primary';
        $optParams = [
          'orderBy' => 'startTime',
          'singleEvents' => TRUE,
          'timeMin' => date('c'),
        ];

        $results = $service->events->listEvents($calendarId, $optParams);
        $events = $results->getItems();

        if (empty($events)) {
          $this->messenger()->addStatus('No upcoming events found.');
        }
        else {
          $_SESSION['york_google_calendar_sync']['york_events'] = $events;
          $form_state->setRedirect('york_google_calendar_sync.events_form');
        }
      }
    }
    else {

      $auth_creds_fid = $config->get('gc_client_credentials');

      // Get credientials.json file URI.
      $credentials = (!empty($auth_creds_fid[0])) ? file_load($auth_creds_fid[0])->getFileUri() : FALSE;

      if (!empty($credentials)) {

        $client = new \Google_Client();
        $client->setApplicationName('Google Calendar API');
        $client->setScopes(\Google_Service_Calendar::CALENDAR_READONLY);
        $client->setAuthConfig($credentials);
        $client->setAccessType('offline');
        $client->setPrompt('select_account consent');

        $this->generateTokenLink($client);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  private function saveSettings(array &$form, FormStateInterface $form_state) {
    $this->config('york_google_calendar_sync.settings')
      ->set('gc_client_credentials', $form_state->getValue('gc_client_credentials'))
      ->set('gc_token_set', $form_state->getValue('gc_token_set'))
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  private function getClient(array &$form, FormStateInterface $form_state, string $auth_code) {
    $config = $this->config('york_google_calendar_sync.settings');

    // Get credentials.json file ID.
    $auth_creds_fid = $config->get('gc_client_credentials');

    // Get credientials.json file URI.
    $credentials = (!empty($auth_creds_fid[0])) ? file_load($auth_creds_fid[0])->getFileUri() : FALSE;

    if (!empty($credentials)) {

      $client = new \Google_Client();
      $client->setApplicationName('Google Calendar API');
      $client->setScopes(\Google_Service_Calendar::CALENDAR_READONLY);
      $client->setAuthConfig($credentials);
      $client->setAccessType('offline');
      $client->setPrompt('select_account consent');

      $tokenPath = 'private://oauth/google_calendar/token.json';
      $has_token = (!file_exists(dirname($tokenPath))) ? TRUE : FALSE;

      if (file_exists($tokenPath)) {
        $accessToken = json_decode(file_get_contents($tokenPath), TRUE);
        $client->setAccessToken($accessToken);
      }

      // If there is no previous token or it's expired.
      if ($client->isAccessTokenExpired()) {

        // Refresh the token if possible, else fetch a new one.
        if ($client->getRefreshToken()) {

          $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());

        }
        else {
          // Request authorization from the user.
          $authUrl = $client->createAuthUrl();

          // Exchange authorization code for an access token.
          $accessToken = $client->fetchAccessTokenWithAuthCode($auth_code);
          $client->setAccessToken($accessToken);

          $this->saveSettings($form, $form_state);

        }
      }
      return $client;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function generateTokenLink(object $client) {

    // Request authorization from the user.
    $authUrl = $client->createAuthUrl();

    $linkText = t('Click here to Grant API Access');
    $authLink = '<a href="' . $authUrl . '">' . $linkText . '</a>';
    $messageText = t('You must grant access to the Google Services API.');
    $this->messenger()->addWarning($messageText . $authLink);

  }

}
