<?php

namespace Drupal\york_live_tweets\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Abraham\TwitterOAuth\TwitterOAuth;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;

/**
 * Provides a block with a simple text.
 *
 * @Block(
 *   id = "york_live_tweets_block",
 *   admin_label = @Translation("York Live Tweets block"),
 * )
 */
class YorkLiveTweetsBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function build() {
    $list_of_tweets = [];
    $quoted_tweets = [];
    $error_message = '';

    $config = \Drupal::config('york_live_tweets.settings');

    $scr_name = $config->get('york_live_tweets_screen_name');
    $max_tweets = $config->get('york_live_tweets_number_of_tweets');
    $api_key = $config->get('york_live_tweets_api_key');
    $api_skey = $config->get('york_live_tweets_api_skey');
    $acc_token = $config->get('york_live_tweets_acc_token');
    $acc_stoken = $config->get('york_live_tweets_acc_stoken');

    $lang_code = \Drupal::languageManager()->getCurrentLanguage()->getId();

    $connection = new TwitterOAuth($api_key, $api_skey, $acc_token, $acc_stoken);
    // Getting account information.
    $account = $connection->get("account/verify_credentials");
    if ($connection->getLastHttpCode() == 200 || !is_numeric($max_tweets)) {
      // Getting tweets.
      $response = $connection->get("search/tweets", [
        "q" => "$scr_name-filter:retweets AND -filter:replies AND from:$scr_name",
        "include_entities" => "true",
        "tweet_mode" => "extended",
        "result_type" => "recent",
        "lang" => $lang_code,
        "count" => ((int) $max_tweets * 5) >= 100 ? 99 : (int) $max_tweets * 5,
      ]);

      if ($connection->getLastHttpCode() == 200) {
        $returnedTweets = $response->statuses;

        // Storing all account data in an object to be used in the template.
        $accountData = (object) [
          'name' => $returnedTweets[0]->user->name,
          'screen_name' => $scr_name,
          'profile_image' => $returnedTweets[0]->user->profile_image_url_https,
          'tweets_url' => 'https://twitter.com/' . $scr_name,
        ];

        // Iterate through tweets that the API responsded with.
        for ($i = 0; $i <= count($returnedTweets) - 1; $i++) {
          $tweet = $returnedTweets[$i];

          // If the list of tweets is of size $max_tweets...
          if (count($list_of_tweets) == $max_tweets) {
            break;
          }
          else {
            // If the tweet is a mention (not actually from York)...
            if (in_array($tweet->id, $quoted_tweets)) {
              continue;
            }

            /* If the tweet contains another tweet (quoted tweet), store the
            quote id so it can be skipped if found later. */
            if ($tweet->is_quote_status) {
              array_push($quoted_tweets, $tweet->quoted_status_id);
            }

            $tweet->formatted_text = $this->formatText($tweet->full_text);
            $tweet->formatted_date = $this->convertDate($tweet->created_at);

            array_push($list_of_tweets, $tweet);
          }
        }

      }
      else {
        $error_message = t("Sorry, we could not retrieve live tweets at this time.");
        $list_of_tweets = [NULL, NULL, NULL, NULL];
      }

    }
    else {
      $error_message = t("Sorry, we could not retrieve live tweets at this time.");
      $list_of_tweets = [NULL, NULL, NULL, NULL];
    }

    return [
      '#theme' => 'york_live_tweets_template',
      '#tweets' => $list_of_tweets,
      '#account' => $accountData,
      '#error_message' => $error_message,
      '#attached' => [
        'library' => [
          'york_live_tweets/styling',
        ],
      ],
      '#cache' => [
        'max-age' => 0,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'access content');
  }

  /**
   * Function to convert date format.
   */
  private function convertDate($date) {
    // Example of $date: Tue Oct 22 18:49:57 +0000 2019 (This comes in GMT time)
    $gmt_tz = new \DateTimeZone('GMT');
    $toronto_tz = new \DateTimeZone('America/Toronto');

    // Converting date from Twitter to Toronto timezone.
    $new_date = new \DateTime($date, $gmt_tz);
    $new_date->setTimezone($toronto_tz);

    // Splitting string by ' ' into an array.
    $date_items = explode(' ', $date);

    // Converting "15:26:00" to "3:26 PM".
    $formatted_time = $new_date->format('g:i A');

    /* This will format "Thu Sep 19 15:26:00 +0000 2019" (which is what
    comes from the API) to "3:26 PM - Sep 19, 2019". */
    return $formatted_time . ' - ' . $date_items[1] . ' ' . $date_items[2] . ', ' . $date_items[5];
  }

  /**
   * Function to format tweet body text.
   */
  private function formatText($text) {
    // Splitting string by ' ' into an array.
    $words = explode(' ', $text);
    $formatted_text = "";

    // Keeping a referrence to the last word.
    $last_word = end($words);

    // Iterate through each word...
    foreach ($words as $word) {
      if (strpos($word, 't.co') !== FALSE && $word == $last_word) {
        /* If current word has t.co in it and it's the last word, skip it
        (the link to the tweet). */
        continue;
      }
      elseif (substr($word, 0, 1) == '@') {
        // If the current word starts with an @.
        $word = '<a href="https://twitter.com/' . substr($word, 1) . '">' . $word . ' </a>';
      }
      elseif (substr($word, 0, 1) == '#') {
        // If the current word starts with an #.
        $word = '<a href="https://twitter.com/hashtag/' . substr($word, 1) . '?src=hash">' . $word . ' </a>';
      }
      elseif ((substr($word, 0, 5) == 'https') || (substr($word, 0, 4) == 'http')) {
        // If the current word starts with an http or https.
        $word = '<a href="' . $word . '">' . $word . ' </a>';
      }

      // Add the formatted word into the formatted_text string.
      $formatted_text = $formatted_text . ' ' . $word;
    }

    return $formatted_text;
  }

}
