<?php

namespace Drupal\york_misc\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Defines a form that configures forms module settings.
 */
class SiteConfigurationForm extends ConfigFormBase {

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a new SiteConfiguration Form.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LanguageManagerInterface $language_manager, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($config_factory);

    $this->languageManager = $language_manager;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('language_manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'yrdsb_workflows_site_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'yrdsb_workflows.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('yrdsb_workflows.settings');

    $form['google_calendar_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Google Calendar API key'),
      '#description' => $this->t('Enter a Google Calendar API key. For ex. AIzaSyDcnW6WejpTOCffshGDDb4neIrXVUA1EAE'),
      '#default_value' => $config->get('google_calendar_api_key'),
    ];
    
    if ($config->get('names_fieldset') == NULL) {
      $default_params = [];
    }
    else {
      $default_params = $config->get('names_fieldset');
      unset($default_params['actions']);
    }
    $name_field = $form_state->get('num_names') ? $form_state->get('num_names') : count($default_params);

    $form['#tree'] = TRUE;

    $form['names_fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Sections'),
      '#prefix' => "<div id='names-fieldset-wrapper'>",
      '#suffix' => '</div>',
    ];

    if (empty($name_field)) {
      $name_field = $form_state->set('num_names', 1);
    }
    else {
         $form_state->set('num_names', $name_field);
    }

    for ($i = 0; $i < $name_field; $i++) {
      $j = $i + 1;
      
      

      $form['names_fieldset'][$i]['link'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Calendar Link '. $j),
        '#maxlength' => 255,
        '#default_value' => count($default_params) ? $default_params[$i]['link'] : "",
        '#size' => 64,
        //'#disabled' => $disable,
      ];
      $form['names_fieldset'][$i]['name'] = [
        '#type' => 'textfield',
        '#default_value' => count($default_params) ? $default_params[$i]['name'] : "",
        '#title' => $this->t('Name'),
        '#maxlength' => 255,
        '#size' => 64,
        //'#disabled' => $disable,
      ];
      $form['names_fieldset'][$i]['color'] = [
        '#type' => 'color_picker',
        '#default_value' => count($default_params) ? $default_params[$i]['color'] : "#000000",
        '#title' => $this->t('Select Color'),
        '#color_values' => '#000000,#ff0000,#00ff00,#0000ff,#ffffff',
        //'#disabled' => $disable,
      ];
      
      if(!empty($default_params) && $default_params[$i]['lock'] == 1) {
          $form['names_fieldset'][$i]['link']['#disabled'] = 'TRUE';
          $form['names_fieldset'][$i]['name']['#disabled'] = 'TRUE';
      }
      
      $form['names_fieldset'][$i]['lock'] = [
        '#type' => 'checkbox',
        '#default_value' => count($default_params) ? $default_params[$i]['lock'] : "",
        '#title' => $this->t('Lock'),
        '#suffix' => "<br><hr><br>",
      ];
    }
    $form['names_fieldset']['actions'] = [
      '#type' => 'actions',
    ];
    $form['names_fieldset']['actions']['add_name'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add one more'),
      '#submit' => ['::addOne'],
      '#ajax' => [
        'callback' => '::addmoreCallback',
        'wrapper' => "names-fieldset-wrapper",
      ],
    ];
    if ($form_state->get('num_names') > 1) {
      $form['names_fieldset']['actions']['remove_name'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove one'),
        '#submit' => ['::removeCallback'],
        '#ajax' => [
          'callback' => '::addmoreCallback',
          'wrapper' => "names-fieldset-wrapper",
        ],
      ];
    }

    $form_state->setCached(FALSE);

    return parent::buildForm($form, $form_state); 
  }
  
  /**
   * Callback for both ajax-enabled buttons.
   *
   * Selects and returns the fieldset with the names in it.
   */
  public function addmoreCallback(array &$form, FormStateInterface $form_state) {
    return $form['names_fieldset'];
  }

  /**
   * Submit handler for the "add-one-more" button.
   *
   * Increments the max counter and causes a rebuild.
   */
  public function addOne(array &$form, FormStateInterface $form_state) {
    $name_field = $form_state->get('num_names');
    $add_button = $name_field + 1;
    $form_state->set('num_names', $add_button);
    $form_state->setRebuild();
  }

  /**
   * Submit handler for the "remove one" button.
   *
   * Decrements the max counter and causes a form rebuild.
   */
  public function removeCallback(array &$form, FormStateInterface $form_state) {
    $name_field = $form_state->get('num_names');
    if ($name_field > 1) {
      $remove_button = $name_field - 1;
      $form_state->set('num_names', $remove_button);
    }
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    $this->config('yrdsb_workflows.settings')
      ->set('names_fieldset', $form_state->getValue('names_fieldset'))
      ->set('google_calendar_api_key', $form_state->getValue('google_calendar_api_key'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
